!***********************************************************************************
!	MODULE:	EulerEquation
!
!	PURPOSE: Contains subroutine EulerEq
!
!	AUTHOR:	Ben Malin
!
!	DATE:	6/2004
!       REVISED: 7/23/2007
!
!***********************************************************************************

module EulerEquation
  
  use Params
  implicit none
  
contains
  
  !***********************************************************************************
  !
  !	SUBROUTINE:  EulerEq
  !
  !	PURPOSE: Specifies system of equations whose solution is the updated
  !               policy function.  These are the equilibrium conditions of the model.
  !
  !	AUTHOR:	 Ben Malin 
  !
  !	DATE:    6/2004
  !     REVISED: 7/23/2007
  !
  !***********************************************************************************

  subroutine EulerEq(pol,EulEq,count)
    
    ! inputs:
    !    pol -- Holds guess of nonlinear equation solver
    !    count -- size of pol vector -- N+1
    !
    ! output:
    !    EulEq -- Values of system of equations evaluated at pol.
    ! 
    ! remarks:
    !    Pol(1) will be the consumption of country 1.
    !    Pol(1+i) for i = {1,..,N} will be tomorrow's capital of country i.
    !
    
    use Params
    implicit none
        
    !Variable Declaration
    integer,intent(in) :: count	
    real(prec),dimension(count),intent(in) :: pol	
    real(prec),dimension(count),intent(out) :: EulEq
    
    integer :: i
    real(prec), dimension(N) :: ctemp
    real(prec), dimension(N) :: ltemp
    real(prec) :: BigNumb = 100
    real(prec) :: IntResult
    
    !Iter1 keeps track of the number of iterations of the nonlinear equation solver
    iter1 = iter1+1
    
    !Make sure guesses for policy function are not out-of-bounds
    do i = 1,N
!       if (pol(1+i) < kmin) then
!          !print*, 'error: capital below kmin in EulerEq', i, iter1, stind
!          !pause
!          ktemp(i) = kmin    !Sets ktemp(i) at the boundary
!       elseif (pol(1+i) > kmax) then
!          !print*, 'error: capital above kmax in EulerEq', i, iter1, stind
!          !pause
!          ktemp(i) = kmax    !Sets ktemp(i) at the boundary
!       else
!          ktemp(i) = pol(1+i)
!       endif
       ktemp(i) = pol(1+i)
    enddo
    if (pol(1) < 0) then
       print*, 'error: consumption negative in EulerEq', 1, iter1
       pause
       ctemp(1) = pol(1)  !Currently no change -- alter if case happens
    else
       ctemp(1) = pol(1)
    endif

    !Compute consumption and labor supply of countries 1 - N.
    do i = 1,N
       ctemp(i) = pol(1) * (Pareto(i)/Pareto(1))**gamma
       ltemp(i) = (Lfoc * exp(z(stind(N+i),i)) * k(stind(i),i)**alpha & 
            & * ctemp(i)**(-xgamma))**(Lelast)
    end do

    !World Budget Constraint
    EulEq(1) = 0
    do i = 1,N
       EulEq(1) = F(k(stind(i),i),ltemp(i),z(stind(N+i),i)) + k(stind(i),i) &
            & - ctemp(i) - pol(1+i) - AdjCost(k(stind(i),i),pol(1+i)) + EulEq(1)
    end do

    !Euler Equation for each country
    call ChebPoly(N, ktemp, unitTtemp(:,:))
    
    do i = 1,N
       call MonIntegrate (i,IntResult,1)
       EulEq(1+i) = betah*IntResult/(1 + AdjCost_ktom(k(stind(i),i),ktemp(i))) & 
            & - Pareto(1)*Uc(ctemp(1))
    end do

    !Penalty if guess for capital is out-of-bounds
    do i = 1,N
!       if (pol(1+i) < kmin) then
!          !print*, EulEq(1+i)
!          EulEq(1+i) = EulEq(1+i) - (pol(1+i) - kmin) * BigNumb
!          !print*, EulEq(1+i), pol(1+i)
!       elseif (pol(1+i) > kmax) then
!          !print*, EulEq(1+i)
!          EulEq(1+i) = EulEq(1+i) + (kmax - pol(1+i)) * BigNumb
!          !print*, EulEq(1+i), pol(1+i)
!       endif
    enddo

  end subroutine EulerEq

end module EulerEquation
