% Decision rules for model A2 with N=4 countries

% Input:    state variables at time t (capital stocks and technology levels)
% Output:   decision variables: dv=[c;l;kp;i;lambda];
%               c:  Nx1 vector of consumption levels
%               l:  Nx1 vector of labor supplies
%               kp: Nx1 vector of next period capital stocks
%               i:  Nx1 vector of investment
%               lambda:  marginal utility of consumption
            


function dv=policies_A2N4(states,dr_coeff_A2N4,kmin,kmax,amin,amax,C_min,C_max,regvec)

k1=states(1);
k2=states(2);
k3=states(3);
k4=states(4);

a1=states(5);
a2=states(6);
a3=states(7);
a4=states(8);

dr_k1=dr_coeff_A2N4(0*end/5+1:1*end/5);
dr_k2=dr_coeff_A2N4(1*end/5+1:2*end/5);
dr_k3=dr_coeff_A2N4(2*end/5+1:3*end/5);
dr_k4=dr_coeff_A2N4(3*end/5+1:4*end/5);
dr_C=dr_coeff_A2N4(4*end/5+1:5*end/5);

% 1) Calibration
N=4;
beta=0.99;
alpha=0.36;
delta=0.025;
sigma=0.01;
rho=0.95;
phi=0.5;
A=(1-beta)/(alpha*beta);
gamma=0.25*ones(1,N);
eta=-0.1*ones(1,N);

% 2) Compute the steady state and the Negishi weights
kss=1;
lss=1;
Kss=N*kss;
Lss=N*lss;
css=A*kss^alpha;
Css=N*css;
lnass=0;
ass=1;
thetass=1;
Yss=A*Kss^alpha*Lss^(1-alpha);


% 3) Compute Negishi weights
tau=zeros(1,N);
for j=1:N
    tau(j)=css^(1/gamma(j));
end
muss=1;

% c) Labor weights
b=zeros(1,N);
for j=1:N
    b(j)=(1-alpha)*A^(1-1/gamma(j));
end

% 4) Scale down states to interval [-1,1] and create Chebyshev polynomials;

sd_k1=2*(k1-kmin)/(kmax-kmin)-1;
sd_k2=2*(k2-kmin)/(kmax-kmin)-1;
sd_k3=2*(k3-kmin)/(kmax-kmin)-1;
sd_k4=2*(k4-kmin)/(kmax-kmin)-1;

sd_a1=2*(a1-amin)/(amax-amin)-1;
sd_a2=2*(a2-amin)/(amax-amin)-1;
sd_a3=2*(a3-amin)/(amax-amin)-1;
sd_a4=2*(a4-amin)/(amax-amin)-1;

Cheby_k1=[1 sd_k1 2*(sd_k1^2)-1];
Cheby_k2=[1 sd_k2 2*(sd_k2^2)-1];
Cheby_k3=[1 sd_k3 2*(sd_k3^2)-1];
Cheby_k4=[1 sd_k4 2*(sd_k4^2)-1];

Cheby_a1=[1 sd_a1 2*(sd_a1^2)-1];
Cheby_a2=[1 sd_a2 2*(sd_a2^2)-1];
Cheby_a3=[1 sd_a3 2*(sd_a3^2)-1];
Cheby_a4=[1 sd_a4 2*(sd_a4^2)-1];

% 5) Create complete polynomial matrix 
NoSt=2*N;
X1=Cheby_k1;
X2=Cheby_k2;
X3=Cheby_k3;
X4=Cheby_k4;

X5=Cheby_a1;
X6=Cheby_a2;
X7=Cheby_a3;
X8=Cheby_a4;

REG=zeros(size(X1,1),1+NoSt+(NoSt-1)*NoSt/2+NoSt);
jjj=1;
REG(:,jjj)=ones(size(X1,1),1);
for i=1:NoSt
    jjj=jjj+1;
    XX=eval(['X' num2str(i)]);
    REG(:,jjj)=XX(:,2);
end
for j=1:(NoSt-1)
    for i=(j+1):NoSt
        jjj=jjj+1;
        XXj=eval(['X' num2str(j)]);
        XXi=eval(['X' num2str(i)]);
        REG(:,jjj)=XXj(:,2).*XXi(:,2);
    end
end
for i=1:NoSt
    jjj=jjj+1;
    XX=eval(['X' num2str(i)]);
    REG(:,jjj)=XX(:,3);
end

% 6) Compute decision variables
    
    % Next period's capital stocks
    kp1=REG*dr_k1;
    kp2=REG*dr_k2;
    kp3=REG*dr_k3;
    kp4=REG*dr_k4;
    kp_vec=[kp1;kp2;kp3;kp4];
    Kp=kp1+kp2+kp3+kp4;
    K=k1+k2+k3+k4;
    
    % Investment
    i1=kp1-(1-delta)*k1;
    i2=kp2-(1-delta)*k2;
    i3=kp3-(1-delta)*k3;
    i4=kp4-(1-delta)*k4;
    i_vec=[i1;i2;i3;i4];
    
    % Capital adjustment costs
    cac1=phi/2*k1.*(kp1./k1-1).^2;
    cac2=phi/2*k2.*(kp2./k2-1).^2;
    cac3=phi/2*k3.*(kp3./k3-1).^2;
    cac4=phi/2*k4.*(kp4./k4-1).^2;
    CAC=cac1+cac2+cac3+cac4;   
    
    % Aggregate consumption
    C=REG*dr_C;   
    
    % Individual consumption 
    % Note: C_min,C_max, and regvec obtain as part of the SGG model
    %       solution; Avoiding non-linear equations solving 
    %       substantially reduces computing times
    
    % Construct complete Chebyshev polynomial matrix to be used for solving
    % for indicudual consumption of country 1
    sd_C=2*(C-C_min)/(C_max-C_min)-1;
    ord=8;
    REGC=ones(1,9);	
    REGC(:,2)=sd_C;
      for i	= 3:9;
            REGC(:,i)	= 2.*sd_C.*REGC(:,i-1)-REGC(:,i-2);
      end;
    c1=REGC*regvec; 
    c2=[tau(1)./tau(2)*c1.^(-1/gamma(1))].^(-gamma(2));
    c3=[tau(1)./tau(3)*c1.^(-1/gamma(1))].^(-gamma(3));
    c4=[tau(1)./tau(4)*c1.^(-1/gamma(1))].^(-gamma(4));
    c_vec=[c1;c2;c3;c4];
      
    % Labor    
    l1=[1/b(1).*[c1.^(-1./gamma(1)).*a1.*(1-alpha).*A.*(k1.^alpha)]].^(1/(alpha-1/eta(1)));
    l2=[1/b(2).*[c2.^(-1./gamma(2)).*a2.*(1-alpha).*A.*(k2.^alpha)]].^(1/(alpha-1/eta(2)));
    l3=[1/b(3).*[c3.^(-1./gamma(3)).*a3.*(1-alpha).*A.*(k3.^alpha)]].^(1/(alpha-1/eta(3)));
    l4=[1/b(4).*[c4.^(-1./gamma(4)).*a4.*(1-alpha).*A.*(k4.^alpha)]].^(1/(alpha-1/eta(4)));
    l_vec=[l1;l2;l3;l4];
    
    % Marginal utility of consumption (of first country)
    lambda=tau(1)*c1^(-1/gamma(1));
    
% 7) Create output
dv=[c_vec;l_vec;kp_vec;i_vec;lambda];