% Decision rules for model A6 with N=2 countries

% Input:    state variables at time t (capital stocks and technology levels)
% Output:   decision variables: dv=[c;l;kp;i;lambda];
%               c:  Nx1 vector of consumption levels
%               l:  Nx1 vector of labor supplies
%               kp: Nx1 vector of next period capital stocks
%               i:  Nx1 vector of investment
%               lambda:  marginal utility of consumption
            
% Note: Model A1 features inelastic labor supply equal to one

function dv=policies_A6N2(states,dr_coeff_A6N2,kmin,kmax,amin,amax,C_min,C_max,regvec)


k1=states(1);
k2=states(2);
a1=states(3);
a2=states(4);

dr_k1=dr_coeff_A6N2(0*end/3+1:1*end/3);
dr_k2=dr_coeff_A6N2(1*end/3+1:2*end/3);
dr_C=dr_coeff_A6N2(2*end/3+1:3*end/3);

% 1) Calibration
N=2;
beta=0.99;
alpha=0.36;
delta=0.025;
sigma=0.01;
rho=0.95;
phi=0.5;
A=(1-beta)/(alpha*beta);
gammalow=0.25;
gammahigh=1;
gamma=[gammalow:(gammahigh-gammalow)/(N-1):gammahigh];
etalow=0.1;
etahigh=1;
eta=-[etalow:(etahigh-etalow)/(N-1):etahigh];

% 2) Compute the steady state and the Negishi weights
kss=1;
lss=1;
Kss=N*kss;
Lss=N*lss;
css=A*kss^alpha;
Css=N*css;
lnass=0;
ass=1;
thetass=1;
Yss=A*Kss^alpha*Lss^(1-alpha);


% 3) Compute Negishi weights
tau=zeros(1,N);
for j=1:N
    tau(j)=css^(1/gamma(j));
end
muss=1;

% c) Labor weights
b=zeros(1,N);
for j=1:N
    b(j)=(1-alpha)*A^(1-1/gamma(j));
end

% 4) Scale down states to interval [-1,1] and create Chebyshev polynomials;

sd_k1=2*(k1-kmin)/(kmax-kmin)-1;
sd_k2=2*(k2-kmin)/(kmax-kmin)-1;
sd_a1=2*(a1-amin)/(amax-amin)-1;
sd_a2=2*(a2-amin)/(amax-amin)-1;

Cheby_k1=[1 sd_k1 2*(sd_k1^2)-1];
Cheby_k2=[1 sd_k2 2*(sd_k2^2)-1];
Cheby_a1=[1 sd_a1 2*(sd_a1^2)-1];
Cheby_a2=[1 sd_a2 2*(sd_a2^2)-1];

% 5) Create complete polynomial matrix 
NoSt=2*N;
X1=Cheby_k1;
X2=Cheby_k2;
X3=Cheby_a1;
X4=Cheby_a2;
REG=zeros(size(X1,1),1+NoSt+(NoSt-1)*NoSt/2+NoSt);
jjj=1;
REG(:,jjj)=ones(size(X1,1),1);
for i=1:NoSt
    jjj=jjj+1;
    XX=eval(['X' num2str(i)]);
    REG(:,jjj)=XX(:,2);
end
for j=1:(NoSt-1)
    for i=(j+1):NoSt
        jjj=jjj+1;
        XXj=eval(['X' num2str(j)]);
        XXi=eval(['X' num2str(i)]);
        REG(:,jjj)=XXj(:,2).*XXi(:,2);
    end
end
for i=1:NoSt
    jjj=jjj+1;
    XX=eval(['X' num2str(i)]);
    REG(:,jjj)=XX(:,3);
end

% 6) Compute decision variables
    
    % Next period's capital stocks
    kp1=REG*dr_k1;
    kp2=REG*dr_k2;
    kp_vec=[kp1;kp2];
    Kp=kp1+kp2;
    K=k1+k2;
    
    % Investment
    i1=kp1-(1-delta)*k1;
    i2=kp2-(1-delta)*k2;
    i_vec=[i1;i2];
    
    % Capital adjustment costs
    cac1=phi/2*k1.*(kp1./k1-1).^2;
    cac2=phi/2*k2.*(kp2./k2-1).^2;
    CAC=cac1+cac2;   
    
    % Aggregate consumption
    C=REG*dr_C;   
    
    % Individual consumption 

    
    % Note: C_min,C_max, and regvec obtain as part of the SGG model
    %       solution; Avoiding non-linear equations solving 
    %       substantially reduces computing times
    
    % Construct complete Chebyshev polynomial matrix to be used for solving
    % for indicudual consumption of country 1
    sd_C=2*(C-C_min)/(C_max-C_min)-1;
    ord=8;
    REGC=ones(1,9);	
    REGC(:,2)=sd_C;
      for i	= 3:9;
            REGC(:,i)	= 2.*sd_C.*REGC(:,i-1)-REGC(:,i-2);
      end;
    c1=REGC*regvec; 
    c2=[tau(1)./tau(2)*c1.^(-1/gamma(1))].^(-gamma(2));
    c_vec=[c1;c2];
      
    % Labor    
    l1=[1/b(1).*[c1.^(-1./gamma(1)).*a1.*(1-alpha).*A.*(k1.^alpha)]].^(1/(alpha-1/eta(1)));
    l2=[1/b(2).*[c2.^(-1./gamma(2)).*a2.*(1-alpha).*A.*(k2.^alpha)]].^(1/(alpha-1/eta(2)));
    l_vec=[l1;l2];
    
    % Marginal utility of consumption (of first country)
    lambda=tau(1)*c1^(-1/gamma(1));
    
% 7) Create output
dv=[c_vec;l_vec;kp_vec;i_vec;lambda];