% Decision rules for model A8 with N=2 countries

% Input:    state variables at time t (capital stocks and technology levels)
% Output:   decision variables: dv=[c;l;kp;i;lambda];
%               c:  Nx1 vector of consumption levels
%               l:  Nx1 vector of labor supplies
%               kp: Nx1 vector of next period capital stocks
%               i:  Nx1 vector of investment
%               lambda:  marginal utility of consumption
            
% Note: Model A1 features inelastic labor supply equal to one

function dv=policies_A8N2(states,dr_coeff_A8N2,kmin,kmax,amin,amax,mumin,mumax,regvec_labor)

k1=states(1);
k2=states(2);
a1=states(3);
a2=states(4);

dr_k1=dr_coeff_A8N2(0*end/3+1:1*end/3);
dr_k2=dr_coeff_A8N2(1*end/3+1:2*end/3);
dr_mu=dr_coeff_A8N2(2*end/3+1:3*end/3);


% 1) Calibration
N=2;
beta=0.99;
alpha=0.36;
delta=0.025;
sigma=0.01;
rho=0.95;
phi=0.5;
A=(1-beta)/(alpha*beta);
gammalow=0.2;
gammahigh=0.4;
gamma=[gammalow:(gammahigh-gammalow)/(N-1):gammahigh];
chilow=0.75;
chihigh=0.9;
chi=[chilow:(chihigh-chilow)/(N-1):chihigh];
muparlow=-0.3;
muparhigh=0.3;
mupar=[muparlow:(muparhigh-muparlow)/(N-1):muparhigh];
Le=2.5;

% 2) Compute the steady state and the Negishi weights
kss=1;
lss=1;
Kss=N*kss;
Lss=N*lss;
css=A*kss^alpha*lss^(1-alpha);
Css=N*css;
lnass=0;
ass=1;
thetass=1;
Yss=N*A;

b=zeros(1,N);
for j=1:N
    b(j)=(1-alpha)*A^(1-1/chi(j))/[(Le-lss)^(-1/chi(j))];
end


% b) Compute Negishi weights
tau=zeros(1,N);
for j=1:N
    tau(j)=[css^(-1/chi(j))*[css^(1-1/chi(j))+b(j)*(Le-lss)^(1-1/chi(j))]^[(1-1/gamma(j))/(1-1/chi(j))-1]]^(-1);
end
muss=1;

% 4) Scale down states to interval [-1,1] and create Chebyshev polynomials;
sd_k1=2*(k1-kmin)/(kmax-kmin)-1;
sd_k2=2*(k2-kmin)/(kmax-kmin)-1;
sd_a1=2*(a1-amin)/(amax-amin)-1;
sd_a2=2*(a2-amin)/(amax-amin)-1;

Cheby_k1=[1 sd_k1 2*(sd_k1^2)-1];
Cheby_k2=[1 sd_k2 2*(sd_k2^2)-1];
Cheby_a1=[1 sd_a1 2*(sd_a1^2)-1];
Cheby_a2=[1 sd_a2 2*(sd_a2^2)-1];

% 5) Create complete polynomial matrix 
NoSt=2*N;
X1=Cheby_k1;
X2=Cheby_k2;
X3=Cheby_a1;
X4=Cheby_a2;
REG=zeros(size(X1,1),1+NoSt+(NoSt-1)*NoSt/2+NoSt);
jjj=1;
REG(:,jjj)=ones(size(X1,1),1);
for i=1:NoSt
    jjj=jjj+1;
    XX=eval(['X' num2str(i)]);
    REG(:,jjj)=XX(:,2);
end
for j=1:(NoSt-1)
    for i=(j+1):NoSt
        jjj=jjj+1;
        XXj=eval(['X' num2str(j)]);
        XXi=eval(['X' num2str(i)]);
        REG(:,jjj)=XXj(:,2).*XXi(:,2);
    end
end
for i=1:NoSt
    jjj=jjj+1;
    XX=eval(['X' num2str(i)]);
    REG(:,jjj)=XX(:,3);
end

% 6) Compute decision variables
    
    % Next period's capital stocks
    kp1=REG*dr_k1;
    kp2=REG*dr_k2;
    kp_vec=[kp1;kp2];
    Kp=kp1+kp2;
    K=k1+k2;
    
    % Investment
    i1=kp1-(1-delta)*k1;
    i2=kp2-(1-delta)*k2;
    i_vec=[i1;i2];
    
    % Capital adjustment costs
    cac1=phi/2*k1.*(kp1./k1-1).^2;
    cac2=phi/2*k2.*(kp2./k2-1).^2;
    CAC=cac1+cac2;   
    
    % lambda=mu
    mu=exp(REG*dr_mu);
    lambda=mu;
    
    % Scale down mu
    sd_mu=2*(mu-mumin)/(mumax-mumin)-1;
       
    % Create Chebyhsev polynomials of order 8 to compute labor
    ord=8;
    
  	REG_K1=ones(1,ord+1);
    REG_K2=ones(1,ord+1);
    REG_A1=ones(1,ord+1);
    REG_A2=ones(1,ord+1);
    REG_MU=ones(1,ord+1);
    
    REG_K1(1,2)=sd_k1;
    REG_K2(1,2)=sd_k2;
    REG_A1(1,2)=sd_a1;
    REG_A2(1,2)=sd_a2;
    REG_MU(1,2)=sd_mu;
    
    for i=3:ord+1;
		REG_K1(1,i)=2.*sd_k1.*REG_K1(:,i-1)-REG_K1(:,i-2);
        REG_K2(1,i)=2.*sd_k2.*REG_K2(:,i-1)-REG_K2(:,i-2);
        REG_A1(1,i)=2.*sd_a1.*REG_A1(:,i-1)-REG_A1(:,i-2);
        REG_A2(1,i)=2.*sd_a2.*REG_A2(:,i-1)-REG_A2(:,i-2);
        REG_MU(1,i)=2.*sd_mu.*REG_MU(:,i-1)-REG_MU(:,i-2);
    end;
    
    counti=0;
    REGMAT1=zeros(size(REG_MU,1),200);
    REGMAT2=zeros(size(REG_MU,1),200);
        for i1=0:ord  
            for i2=0:[ord-i1];        
                for i3=0:[ord-i1-i2];
                    counti=counti+1;
                    REGMAT1(:,counti)=REG_MU(:,i1+1).*REG_A1(:,i2+1).*REG_K1(:,i3+1);
                    REGMAT2(:,counti)=REG_MU(:,i1+1).*REG_A2(:,i2+1).*REG_K2(:,i3+1);
                end
            end
        end

    REGMAT1=REGMAT1(:,1:counti);
    REGMAT2=REGMAT2(:,1:counti);
    
    l1=REGMAT1*regvec_labor(:,1);
    l2=REGMAT2*regvec_labor(:,2);
    l_vec=[l1;l2];
    
    % Consumption
        c1=[[b(1)*(Le-l1).^(-1/chi(1))]./[a1.*(1-alpha).*A.*(l1.^(mupar(1)-1)).*...
            (alpha.*k1.^(mupar(1))+(1-alpha)*l1.^mupar(1)).^(1/mupar(1)-1)]].^(-chi(1));
        c2=[[b(2)*(Le-l2).^(-1/chi(2))]./[a2.*(1-alpha).*A.*(l2.^(mupar(2)-1)).*...
            (alpha.*k2.^(mupar(2))+(1-alpha)*l2.^mupar(2)).^(1/mupar(2)-1)]].^(-chi(2));
        C= c1+c2;
    c_vec=[c1;c2];
 
% 7) Create output
dv=[c_vec;l_vec;kp_vec;i_vec;lambda];
