!****************************************************************************
!
!	MODULE:  Params
!
!	PURPOSE:  Declares and initializes global variables
!			 
!	AUTHOR:	 Ben Malin 
!
!	DATE:  10/2007
!****************************************************************************

module Params
  
  implicit none
  
  integer,parameter :: prec = selected_real_kind(15,307)
  real(prec),parameter :: pie = 3.1415926535897932
  
  !------------------------------
  !Number of 'state' variables
  !------------------------------
  integer,parameter :: N = COUNTRIES 	   !Number of countries  
  integer,parameter :: d = 2*COUNTRIES !2*N!   !Dimension of state space (capital and tech shock)
  integer,parameter :: q = 2*COUNTRIES+2 !d+2!   !In smolyak interpolation algorithm, lambda = q-d

  !----------------------
  !Parameters of model
  !----------------------
  real(prec),parameter :: gamma_min = 0.25
  real(prec),parameter :: gamma_max = 1.0
  real(prec),dimension(N) :: gamma     !Risk aversion parameter
  real(prec),dimension(N) :: xgamma    !Intertemporal Elasticity of Substitution
  real(prec),parameter :: Le = 2.5     !Time endowment
  real(prec),dimension(N) :: psi       !Parameter that governs level of labor supply
  real(prec),parameter :: betah = 0.99	!Subjective discount factor
  real(prec),parameter :: alpha = 0.36	!Capital Share
  real(prec),parameter :: delta = 0.025	!Depreciation
  real(prec) :: kappa = 0.5		!Adjustment Cost parameter  ("phi" in paper)
  
  real(prec) :: A_tfp = (1-betah)/(alpha*betah)	 !Total Factor Productivity
  real(prec),parameter :: sigma = 0.01	 !Variance of tech. shock
  real(prec),parameter :: rho = 0.95     !Persistence of tech. shock
  
  real(prec),dimension(N) :: Pareto    !Pareto weights for each of N countries 

  real(prec),dimension(N) :: Lfoc !Parameter in intratemporal first order condition
  real(prec),dimension(N) :: psigam1, psigam2  !Utility function parameters
  
  !--------------------------------
  !Variables for grid subroutine
  !--------------------------------
  real(prec) :: kss	! deterministic steady state capital stock
  real(prec),dimension(N) :: kmax	! used to put upper bound on capital grid
  real(prec),dimension(N) :: kmin	! used to put lower bound on capital grid

  real(prec) :: lss	! deterministic steady state labor supply
  
  real(prec) :: zmax	! largest positive shock is e^(zmax)
  real(prec) :: zmin	! largest negative shock is e^(zmin)
  
  !--------------------------------------------
  !Weights used in formulas for interpolation 
  !--------------------------------------------
  integer :: coc1	!Used in weighted sum of cheb polynomials in Smolyak interpolation
  integer :: coc2	!coc1 is (d-1) choose 1 and coc2 is (d-1) choose 2
  
  !--------------------------------------------------------
  !Variables that hold coefficients of policy functions 
  !--------------------------------------------------------
  real(prec),dimension(d,2**(q-d)+1,N) :: CoeffsCons3	!Coeffs for 4th order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d)+1,N) :: CoeffsCap3    !Coeffs for 4th order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d)+1,N) :: CoeffsLab3    !Coeffs for 4th order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d-1)+1,N) :: CoeffsCons1	!Coeffs for 2nd order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d-1)+1,N) :: CoeffsCap1	!Coeffs for 2nd order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d-1)+1,N) :: CoeffsLab1	!Coeffs for 2nd order polynomials for each of d dims
  real(prec),dimension(d-1,d,2**(q-d-1)+1,2**(q-d-1)+1,N) :: CoeffsCons2 !Coeffs for 2nd order tensor prods
  real(prec),dimension(d-1,d,2**(q-d-1)+1,2**(q-d-1)+1,N) :: CoeffsCap2	 !Coeffs for 2nd order tensor prods
  real(prec),dimension(d-1,d,2**(q-d-1)+1,2**(q-d-1)+1,N) :: CoeffsLab2	 !Coeffs for 2nd order tensor prods
  real(prec),dimension(N) :: yCons0		!Policy function (consumption) evaluated at point0
  real(prec),dimension(N) :: yCap0		!Policy function (capital) evaluated at point0
  real(prec),dimension(N) :: yLab0		!Policy function (labor) evaluated at point0
  
  !--------------------------------
  !General global variables
  !--------------------------------
  integer :: FOCi
  real(prec),dimension(N) :: SimConsumption, SimLabor, SimCapTom, SimOutput
  real(prec),dimension(N) :: SimCapital, shock
  real(prec),dimension(2*N) :: instate
  real(prec),dimension(4*N) :: Results
  
contains
  
  !*************************
  !** Primative Functions **
  !*************************
  
  !-------------------------------------------------------
  ! function Ucc
  !
  !	2nd Derivative wrt consumption of a Cobb-Douglas 
  !     utility function over consumption and leisure.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  !             i -- integer, country index
  ! Returns:	real, Ucc
  !
  !-------------------------------------------------------
  
  function Ucc(c,lab,i)
    implicit none
    real(prec) :: Ucc
    real(prec), intent(in) :: c,lab
    integer, intent(in) :: i
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Ucc = psi(i) * (psigam1(i) - 1.0) * c**(psigam1(i) - 2.0) * (Le - lab)**psigam2(i)
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Ucc, lab = ', lab
       pause
       Ucc = psi(i) * (psigam1(i) - 1.0) * c**(psigam1(i) - 2.0) * (Le - epsilon)**psigam2(i)       
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Ucc, lab = ', lab
       pause
       Ucc = psi(i) * (psigam1(i) - 1.0) * c**(psigam1(i) - 2.0) * (Le - L_upper)**psigam2(i)       
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Ucc, c = ', c
       print*, 'lab is below lower bound in Ucc, lab = ', lab
       pause
       Ucc = psi(i) * (psigam1(i) - 1.0) * epsilon**(psigam1(i) - 2.0) * (Le - epsilon)**psigam2(i)      
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Ucc, c = ', c
       print*, 'lab is above upper bound in Ucc, lab = ', lab
       pause
       Ucc = psi(i) * (psigam1(i) - 1.0) * epsilon**(psigam1(i) - 2.0) * (Le - L_upper)**psigam2(i)       
    else
       print*, 'c is below lower bound in the function Ucc, c = ', c	
       pause
       Ucc = psi(i) * (psigam1(i) - 1.0) * epsilon**(psigam1(i) - 2.0) * (Le - lab)**psigam2(i)       
    end if
    
  end function Ucc
  
  !-------------------------------------------------------
  ! function Ucl
  !
  !	2nd Derivative of a Cobb-Douglas utility function 
  !     over consumption and leisure wrt consumption and 
  !     labor.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  !             i -- integer, country index
  ! Returns:	real, Ucl
  !
  !-------------------------------------------------------
  
  function Ucl(c,lab,i)
    implicit none
    real(prec) :: Ucl
    real(prec), intent(in) :: c,lab
    integer, intent(in) :: i
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Ucl = -1.0 * psi(i) * psigam2(i) * c**(psigam1(i) - 1.0) * (Le - lab)**(psigam2(i) - 1.0)
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Ucl, lab = ', lab
       pause
       Ucl = -1.0 * psi(i) * psigam2(i) * c**(psigam1(i) - 1.0) * (Le - epsilon)**(psigam2(i) - 1.0)
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Ucl, lab = ', lab
       pause
       Ucl = -1.0 * psi(i) * psigam2(i) * c**(psigam1(i) - 1.0) * (Le - L_upper)**(psigam2(i) - 1.0)
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Ucl, c = ', c
       print*, 'lab is below lower bound in Ucl, lab = ', lab
       pause
       Ucl = -1.0 * psi(i) * psigam2(i) * epsilon**(psigam1(i) - 1.0) * (Le - epsilon)**(psigam2(i) - 1.0)
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Ucl, c = ', c
       print*, 'lab is above upper bound in Ucl, lab = ', lab
       pause
       Ucl = -1.0 * psi(i) * psigam2(i) * epsilon**(psigam1(i) - 1.0) * (Le - L_upper)**(psigam2(i) - 1.0)
    else
       print*, 'c is below lower bound in the function Ucl, c = ', c	
       pause
       Ucl = -1.0 * psi(i) * psigam2(i) * epsilon**(psigam1(i) - 1.0) * (Le - lab)**(psigam2(i) - 1.0)
    end if
    
  end function Ucl
  
  !-------------------------------------------------------
  ! function Ull
  !
  !	2nd Derivative a Cobb-Douglas utility function 
  !     over consumption and leisure wrt labor.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  !             i -- integer, country index
  ! Returns:	real, Ull
  !
  !-------------------------------------------------------
  
  function Ull(c,lab,i)
    implicit none
    real(prec) :: Ull
    real(prec), intent(in) :: c,lab
    integer, intent(in) :: i
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Ull = (1.0 - psi(i)) * (psigam2(i) - 1.0) * c**psigam1(i) * (Le - lab)**(psigam2(i) - 2.0)
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Ull, lab = ', lab
       pause
       Ull = (1.0 - psi(i)) * (psigam2(i) - 1.0) * c**psigam1(i) * (Le - epsilon)**(psigam2(i) - 2.0)
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Ull, lab = ', lab
       pause
       Ull = (1.0 - psi(i)) * (psigam2(i) - 1.0) * c**psigam1(i) * (Le - L_upper)**(psigam2(i) - 2.0)
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Ull, c = ', c
       print*, 'lab is below lower bound in Ull, lab = ', lab
       pause
       Ull = (1.0 - psi(i)) * (psigam2(i) - 1.0) * epsilon**psigam1(i) * (Le - epsilon)**(psigam2(i) - 2.0)
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Ull, c = ', c
       print*, 'lab is above upper bound in Ull, lab = ', lab
       pause
       Ull = (1.0 - psi(i)) * (psigam2(i) - 1.0) * epsilon**psigam1(i) * (Le - L_upper)**(psigam2(i) - 2.0)
    else
       print*, 'c is below lower bound in the function Ull, c = ', c	
       pause
       Ull = (1.0 - psi(i)) * (psigam2(i) - 1.0) * epsilon**psigam1(i) * (Le - lab)**(psigam2(i) - 2.0)
    end if
    
  end function Ull
 
  !-------------------------------------------------------
  ! function Uc
  !
  !	1st Derivative wrt consumption of a Cobb-Douglas 
  !     utility function over consumption and leisure.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  !             i -- integer, country index
  ! Returns:	real, marginal utility of consumption
  !
  !-------------------------------------------------------
  
  function Uc(c,lab,i)
    implicit none
    real(prec) :: Uc
    real(prec), intent(in) :: c,lab
    integer, intent(in) :: i
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Uc = psi(i) * c**(psigam1(i) - 1.0) * (Le - lab)**psigam2(i)
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Uc, lab = ', lab
!       pause
       Uc = psi(i) * c**(psigam1(i) - 1.0) * (Le - epsilon)**psigam2(i) + Ucl(c,epsilon,i)*(lab-epsilon)
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Uc, lab = ', lab
!       pause
       Uc = psi(i) * c**(psigam1(i) - 1.0) * (Le - L_upper)**psigam2(i) + Ucl(c,L_upper,i)*(lab-L_upper)
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Uc, c = ', c
       print*, 'lab is below lower bound in Uc, lab = ', lab
!       pause
       Uc = psi(i) * epsilon**(psigam1(i) - 1.0) * (Le - epsilon)**psigam2(i) &
            & + Ucl(epsilon,epsilon,i)*(lab-epsilon) + Ucc(epsilon,epsilon,i)*(c-epsilon)        
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Uc, c = ', c
       print*, 'lab is above upper bound in Uc, lab = ', lab
!       pause
       Uc = psi(i) * epsilon**(psigam1(i) - 1.0) * (Le - L_upper)**psigam2(i) &
            & + Ucl(epsilon,L_upper,i)*(lab-L_upper) + Ucc(epsilon,L_upper,i)*(c-epsilon)        
    else
       print*, 'c is below lower bound in the function Uc, c = ', c	
!       pause
       Uc = psi(i) * epsilon**(psigam1(i) - 1.0) * (Le - lab)**psigam2(i) &
            & + Ucc(epsilon,lab,i)*(c-epsilon)        
    end if
    
  end function Uc
  
  !--------------------------------------------------------
  ! function UlUc
  !
  !	Marginal Rate of Substitution between consumption 
  !     and labor
  !
  ! Inputs:	c -- real, consumption
  !             lab -- real, labor supply
  !             i -- integer, country index
  ! Returns:	real, marginal rate of substitution
  !
  !--------------------------------------------------------
  
  function UlUc(c,lab,i)
    implicit none
    
    real(prec) :: UlUc	
    real(prec), intent(in) :: c
    real(prec), intent(in) :: lab
    integer, intent(in) :: i 
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       UlUc = (psi(i) - 1.0) * c / (psi(i)*(Le - lab))
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in UlUc, lab = ', lab
       pause
       UlUc = (psi(i) - 1.0)/psi(i) *( c / (Le - epsilon) + c / (Le-epsilon)**2 * (lab - epsilon))
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in UlUc, lab = ', lab
       pause
       UlUc = (psi(i) - 1.0)/psi(i) *( c / (Le - L_upper) + c / (Le-L_upper)**2 * (lab - L_upper))
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in UlUc, c = ', c
       print*, 'lab is below lower bound in UlUc, lab = ', lab
       pause
       UlUc = (psi(i) - 1.0)/psi(i) * (c/(Le - epsilon) + epsilon / (Le-epsilon)**2 * (lab - epsilon))
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in UlUc, c = ', c
       print*, 'lab is above upper bound in UlUc, lab = ', lab
       pause
       UlUc = (psi(i) - 1.0)/psi(i) * (c/(Le - L_upper) + epsilon / (Le-L_upper)**2 * (lab - L_upper))
    else
       print*, 'c is below lower bound in the function UlUc, c = ', c	
       pause
       UlUc = (psi(i) - 1.0)/psi(i) * c/(Le - lab)
    end if
    
  end function UlUc

  !------------------------------------------------------------------------------------
  ! function F
  !
  !	Production Function - Takes country-specific capital stock and 
  !	technology shock and returns output.
  !
  ! Inputs:	capital -- real, capital level
  !             lab -- real, labor supply
  !	        sh -- real, technological shock   
  ! Returns:	real, output
  !
  !------------------------------------------------------------------------------------
  
  function F(capital,lab,sh)
    implicit none
    real(prec) :: F
    real(prec), intent(in) :: capital,lab,sh
    real(prec) :: epsilon
    
    epsilon = (10.0)**(-6)
    
    if (capital >= epsilon .and. lab >= epsilon) then
       F =  A_tfp * exp(sh) * capital**alpha * lab**(1.0-alpha)
    elseif (capital < epsilon .and. lab >= epsilon) then
       print*, "Error - Capital Stock is negative in F, cap = ", capital 
       pause
       F = 0
    elseif (capital >= epsilon .and. lab < epsilon) then
       print*, "Error - Labor is negative in F, lab = ", lab 
       pause
       F = 0
    else
       print*, "Error - Capital Stock is negative in F, cap = ", capital 
       print*, "Error - Labor is negative in F, lab = ", lab 
       pause
       F = 0
    end if
    
  end function F
  
  !----------------------------------------------------------------------------
  ! function Fk
  !
  !	Marginal Product of Capital - Takes country specific capital stock and 
  !	technology shock and returns mpk.
  !
  ! Inputs:	capital -- real, capital level
  !             lab -- real, labor supply
  !		sh -- real, technological shock   
  ! Returns:	real, mpk
  !
  !----------------------------------------------------------------------------
  
  function Fk(capital,lab,sh)
    implicit none
    real(prec) :: Fk
    real(prec), intent(in) :: capital,lab,sh
    real(prec) :: epsilon
    
    epsilon = (10.0)**(-6)
    
    if (capital >= epsilon .and. lab >= epsilon) then
       Fk =  alpha * A_tfp * exp(sh) * (capital/lab)**(alpha-1.0) 
    elseif (capital < epsilon .and. lab >= epsilon) then
       print*, "Error - Capital Stock is negative in Fk, cap = ", capital  
       pause
       Fk =  alpha * A_tfp * exp(sh) * epsilon**(alpha-1.0) * lab**(1.0-alpha)
    elseif (capital >= epsilon .and. lab < epsilon) then
       print*, "Error - Labor Supply is negative in Fk, lab = ", lab  
       pause
       Fk =  alpha * A_tfp * exp(sh) * capital**(alpha-1.0) * epsilon**(1.0-alpha)
    else
       print*, "Error - Capital Stock is negative in Fk, cap = ", capital  
       print*, "Error - Labor Supply is negative in Fk, lab = ", lab  
       pause
       Fk =  alpha * A_tfp * exp(sh) * epsilon**(alpha-1.0) * epsilon**(1.0-alpha)
    end if
    
  end function Fk
  
  !----------------------------------------------------------------------------
  ! function Fl
  !
  !	Marginal Product of Labor - Takes country specific capital stock and 
  !	technology shock and returns mpl.
  !
  ! Inputs:	capital -- real, capital level
  !             lab -- real, labor supply
  !		sh -- real, technological shock   
  ! Returns:	real, mpl
  !
  !----------------------------------------------------------------------------
  
  function Fl(capital,lab,sh)
    implicit none
    real(prec) :: Fl
    real(prec), intent(in) :: capital,lab,sh
    real(prec) :: epsilon
    
    epsilon = (10.0)**(-6)
    
    if (capital >= epsilon .and. lab >= epsilon) then
       Fl = (1.0-alpha) * A_tfp * exp(sh) * (capital/lab)**(alpha) 
    elseif (capital < epsilon .and. lab >= epsilon) then
       print*, "Error - Capital Stock is negative in Fl, cap = ", capital  
       pause
       Fl = (1.0-alpha) * A_tfp * exp(sh) * epsilon**(alpha) * lab**(-alpha)
    elseif (capital >= epsilon .and. lab < epsilon) then
       print*, "Error - Labor Supply is negative in Fl, lab = ", lab  
       pause
       Fl = (1.0-alpha) * A_tfp * exp(sh) * capital**(alpha) * epsilon**(-alpha)
    else
       print*, "Error - Capital Stock is negative in Fl, cap = ", capital  
       print*, "Error - Labor Supply is negative in Fl, lab = ", lab  
       pause
       Fl = (1.0-alpha) * A_tfp * exp(sh) * epsilon**(alpha) * epsilon**(-alpha)
    end if
    
  end function Fl
  
end module Params
