!****************************************************************************
!
!	MODULE:  Params
!
!	PURPOSE:  Declares and initializes global variables
!			 
!	AUTHOR:	 Ben Malin 
!
!	DATE:  11/2007
!****************************************************************************

module Params
  
  implicit none
  
  integer,parameter :: prec = selected_real_kind(15,307)
  real(prec),parameter :: pie = 3.1415926535897932
  
  !------------------------------
  !Number of 'state' variables
  !------------------------------
  integer,parameter :: N = COUNTRIES 	   !Number of countries  
  integer,parameter :: d = 2*COUNTRIES !2*N!   !Dimension of state space (capital and tech shock)
  integer,parameter :: q = 2*COUNTRIES+2 !d+2!   !In smolyak interpolation algorithm, lambda = q-d

  !----------------------
  !Parameters of model
  !----------------------
  real(prec) :: gamma = 0.25	!Risk aversion parameter
  real(prec) :: xgamma          !IES = 1/gamma (set in ParamUpdate)
  real(prec),parameter :: Le = 2.5   !Time endowment
  real(prec),parameter :: chi = 0.83	!elast. of subst. between cons & leisure 
  real(prec) :: xchi = 1.0/chi
  real(prec) :: ychi   !(set in ParamUpdate)
  real(prec) :: chigam !(set in ParamUpdate)
  real(prec),parameter :: betah = 0.99	!Subjective discount factor
  real(prec),parameter :: alpha = 0.36	!Capital Share
  real(prec),parameter :: muh = -0.20	!production function parameter 
  real(prec),parameter :: delta = 0.025	!Depreciation
  real(prec) :: kappa = 0.5		!Adjustment Cost parameter  ("phi" in paper)
  
  real(prec) :: A_tfp = (1-betah)/(alpha*betah)	 !Total Factor Productivity
  real(prec) :: b    !sets level of labor supply (set in ParamUpdate)
  real(prec),parameter :: sigma = 0.01	 !Variance of tech. shock
  real(prec),parameter :: rho = 0.95     !Persistence of tech. shock
  
  real(prec),dimension(N) :: Pareto    !Pareto weights for each of N countries (set in Grid)

  real(prec) :: Lfoc !Parameter in intratemporal first order condition
  
  !--------------------------------
  !Variables for grid subroutine
  !--------------------------------
  real(prec) :: kss	! deterministic steady state capital stock
  real(prec) :: kmax	! used to put upper bound on capital grid
  real(prec) :: kmin	! used to put lower bound on capital grid

  real(prec) :: lss	! deterministic steady state labor supply
  
  real(prec) :: zmax	! largest positive shock is e^(zmax)
  real(prec) :: zmin	! largest negative shock is e^(zmin)
  
  !--------------------------------------------
  !Weights used in formulas for interpolation 
  !--------------------------------------------
  integer :: coc1	!Used in weighted sum of cheb polynomials in Smolyak interpolation
  integer :: coc2	!coc1 is (d-1) choose 1 and coc2 is (d-1) choose 2
  
  !--------------------------------------------------------
  !Variables that hold coefficients of policy functions 
  !--------------------------------------------------------
  real(prec),dimension(d,2**(q-d)+1,N) :: CoeffsCons3	!Coeffs for 4th order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d)+1,N) :: CoeffsCap3    !Coeffs for 4th order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d)+1,N) :: CoeffsLab3    !Coeffs for 4th order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d-1)+1,N) :: CoeffsCons1	!Coeffs for 2nd order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d-1)+1,N) :: CoeffsCap1	!Coeffs for 2nd order polynomials for each of d dims
  real(prec),dimension(d,2**(q-d-1)+1,N) :: CoeffsLab1	!Coeffs for 2nd order polynomials for each of d dims
  real(prec),dimension(d-1,d,2**(q-d-1)+1,2**(q-d-1)+1,N) :: CoeffsCons2 !Coeffs for 2nd order tensor prods
  real(prec),dimension(d-1,d,2**(q-d-1)+1,2**(q-d-1)+1,N) :: CoeffsCap2	 !Coeffs for 2nd order tensor prods
  real(prec),dimension(d-1,d,2**(q-d-1)+1,2**(q-d-1)+1,N) :: CoeffsLab2	 !Coeffs for 2nd order tensor prods
  real(prec),dimension(N) :: yCons0		!Policy function (consumption) evaluated at point0
  real(prec),dimension(N) :: yCap0		!Policy function (capital) evaluated at point0
  real(prec),dimension(N) :: yLab0		!Policy function (labor) evaluated at point0
  
  !--------------------------------
  !General global variables
  !--------------------------------
  integer :: FOCi
  real(prec),dimension(N) :: SimConsumption, SimLabor, SimCapTom, SimOutput
  real(prec),dimension(N) :: SimCapital, shock
  real(prec),dimension(2*N) :: instate
  real(prec),dimension(4*N) :: Results
  
contains
  
  !*************************
  !** Primative Functions **
  !*************************
  
  !-------------------------------------------------------
  ! function Ucc
  !
  !	2nd Derivative wrt consumption of CES 
  !     utility function over consumption and leisure.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  ! Returns:	real, Ucc
  !
  !-------------------------------------------------------
  
  function Ucc(c,lab)
    implicit none
    real(prec) :: Ucc
    real(prec), intent(in) :: c,lab
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Ucc = chigam * (c**ychi + b*(Le - lab)**ychi)**(chigam/ychi - 1.0) / c**(2.0*xchi) &
            & - xchi * (c**ychi + b*(Le - lab)**ychi)**(chigam/ychi) / c**(xchi + 1.0)
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Ucc, lab = ', lab
       pause
       Ucc = chigam * (c**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi - 1.0) / c**(2.0*xchi) &
            & - xchi * (c**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi) / c**(xchi + 1.0)
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Ucc, lab = ', lab
       pause
       Ucc = chigam * (c**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi - 1.0) / c**(2.0*xchi) &
            & - xchi * (c**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi) / c**(xchi + 1.0)
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Ucc, c = ', c
       print*, 'lab is below lower bound in Ucc, lab = ', lab
       pause
       Ucc = chigam * (epsilon**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi - 1.0) / epsilon**(2.0*xchi) &
            & - xchi * (epsilon**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi) / epsilon**(xchi + 1.0)
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Ucc, c = ', c
       print*, 'lab is above upper bound in Ucc, lab = ', lab
       pause
       Ucc = chigam * (epsilon**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi - 1.0) / epsilon**(2.0*xchi) &
            & - xchi * (epsilon**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi) / epsilon**(xchi + 1.0)
    else
       print*, 'c is below lower bound in the function Ucc, c = ', c	
       pause
       Ucc = chigam * (epsilon**ychi + b*(Le - lab)**ychi)**(chigam/ychi - 1.0) / epsilon**(2.0*xchi) &
            & - xchi * (epsilon**ychi + b*(Le - lab)**ychi)**(chigam/ychi) / epsilon**(xchi + 1.0)
    end if
    
  end function Ucc
  
  !-------------------------------------------------------
  ! function Ucl
  !
  !	2nd Derivative of CES utility function 
  !     over consumption and leisure wrt consumption and 
  !     labor.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  ! Returns:	real, Ucl
  !
  !-------------------------------------------------------
  
  function Ucl(c,lab)
    implicit none
    real(prec) :: Ucl
    real(prec), intent(in) :: c,lab
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Ucl = chigam * (c**ychi + b*(Le - lab)**ychi)**(chigam/ychi - 1.0) / c**xchi &
            & * -1.0 * b / (Le-lab)**xchi
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Ucl, lab = ', lab
       pause
       Ucl = chigam * (c**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi - 1.0) / c**xchi &
            & * -1.0 * b / (Le-epsilon)**xchi
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Ucl, lab = ', lab
       pause
       Ucl = chigam * (c**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi - 1.0) / c**xchi &
            & * -1.0 * b / (Le-L_upper)**xchi
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Ucl, c = ', c
       print*, 'lab is below lower bound in Ucl, lab = ', lab
       pause
       Ucl = chigam * (epsilon**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi - 1.0) / epsilon**xchi &
            & * -1.0 * b / (Le-epsilon)**xchi
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Ucl, c = ', c
       print*, 'lab is above upper bound in Ucl, lab = ', lab
       pause
       Ucl = chigam * (epsilon**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi - 1.0) / epsilon**xchi &
            & * -1.0 * b / (Le-L_upper)**xchi
    else
       print*, 'c is below lower bound in the function Ucl, c = ', c	
       pause
       Ucl = chigam * (epsilon**ychi + b*(Le - lab)**ychi)**(chigam/ychi - 1.0) / epsilon**xchi &
            & * -1.0 * b / (Le-lab)**xchi
    end if
    
  end function Ucl
  
  !-------------------------------------------------------
  ! function Ull
  !
  !	2nd Derivative of a CES utility function 
  !     over consumption and leisure wrt labor.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  ! Returns:	real, Ull
  !
  !-------------------------------------------------------
  
  function Ull(c,lab)
    implicit none
    real(prec) :: Ull
    real(prec), intent(in) :: c,lab
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Ull = chigam * (c**ychi + b*(Le - lab)**ychi)**(chigam/ychi - 1.0) * b**2.0 / &
            & (Le - lab)**(2.0*xchi) - xchi * b / (Le-lab)**(xchi + 1.0) &
            & * (c**ychi + b*(Le - lab)**ychi)**(chigam/ychi)
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Ull, lab = ', lab
       pause
       Ull = chigam * (c**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi - 1.0) * b**2 &
            & / (Le - epsilon)**(2.0*xchi) - xchi * b / (Le-epsilon)**(xchi + 1.0) &
            & * (c**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi)
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Ull, lab = ', lab
       pause
       Ull = chigam * (c**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi - 1.0) * b**2 &
            & / (Le - L_upper)**(2.0*xchi) - xchi * b / (Le-L_upper)**(xchi + 1.0) &
            & * (c**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi)
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Ull, c = ', c
       print*, 'lab is below lower bound in Ull, lab = ', lab
       pause
       Ull = chigam * (epsilon**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi - 1.0) * b**2 &
            & / (Le - epsilon)**(2.0*xchi) - xchi * b / (Le-epsilon)**(xchi + 1.0) &
            & * (epsilon**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi)
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Ull, c = ', c
       print*, 'lab is above upper bound in Ull, lab = ', lab
       pause
       Ull = chigam * (epsilon**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi - 1.0) * b**2 &
            & / (Le -  L_upper)**(2.0*xchi) - xchi * b / (Le- L_upper)**(xchi + 1.0) &
            & * (epsilon**ychi + b*(Le -  L_upper)**ychi)**(chigam/ychi)
    else
       print*, 'c is below lower bound in the function Ull, c = ', c	
       pause
       Ull = chigam * (epsilon**ychi + b*(Le - lab)**ychi)**(chigam/ychi - 1.0) * b**2 &
            & / (Le - lab)**(2.0*xchi) - xchi * b / (Le-lab)**(xchi + 1.0) &
            & * (epsilon**ychi + b*(Le - lab)**ychi)**(chigam/ychi)
    end if
    
  end function Ull
 
  !-------------------------------------------------------
  ! function Uc
  !
  !	1st Derivative wrt consumption of a CES 
  !     utility function over consumption and leisure.
  !
  ! Inputs:	c -- real, consumption 
  !             lab -- real, labor
  ! Returns:	real, marginal utility of consumption
  !
  !-------------------------------------------------------
  
  function Uc(c,lab)
    implicit none
    real(prec) :: Uc
    real(prec), intent(in) :: c,lab
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       Uc = (c**ychi + b*(Le - lab)**ychi)**(chigam/ychi) / c**xchi
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in Uc, lab = ', lab
       pause
       Uc = (c**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi) / c**xchi &
            & + Ucl(c,epsilon)*(lab-epsilon)       
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in Uc, lab = ', lab
       pause
       Uc = (c**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi) / c**xchi &
            & + Ucl(c,L_upper)*(lab-L_upper)       
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in Uc, c = ', c
       print*, 'lab is below lower bound in Uc, lab = ', lab
       pause
       Uc = (epsilon**ychi + b*(Le - epsilon)**ychi)**(chigam/ychi) / epsilon**xchi &
            & + Ucl(epsilon,epsilon)*(lab-epsilon) + Ucc(epsilon,epsilon)*(c-epsilon)        
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in Uc, c = ', c
       print*, 'lab is above upper bound in Uc, lab = ', lab
       pause
       Uc = (epsilon**ychi + b*(Le - L_upper)**ychi)**(chigam/ychi) / epsilon**xchi &
            & + Ucl(epsilon,L_upper)*(lab-L_upper) + Ucc(epsilon,L_upper)*(c-epsilon)        
    else
       print*, 'c is below lower bound in the function Uc, c = ', c	
       pause
       Uc = (epsilon**ychi + b*(Le - lab)**ychi)**(chigam/ychi) / epsilon**xchi &
            & + Ucc(epsilon,lab)*(c-epsilon)        
    end if
    
  end function Uc

  !--------------------------------------------------------
  ! function UlUc
  !
  !	Marginal Rate of Substitution between consumption 
  !     and labor
  !
  ! Inputs:	c -- real, consumption
  !             lab -- real, labor supply
  ! Returns:	real, marginal rate of substitution
  !
  !--------------------------------------------------------
  
  function UlUc(c,lab)
    implicit none
    
    real(prec) :: UlUc	
    real(prec), intent(in) :: c
    real(prec), intent(in) :: lab
    real(prec) :: epsilon, L_upper
    
    epsilon = (10.0)**(-6)
    L_upper = Le - epsilon
    
    if (c >= epsilon .and. lab >= epsilon .and. lab <= L_upper) then
       UlUc = -1.0 * b * (c / (Le - lab))**xchi
    elseif (c >= epsilon .and. lab < epsilon) then
       print*, 'lab is below lower bound in UlUc, lab = ', lab
       pause
       UlUc = -1.0 * b * (c / (Le - epsilon))**xchi
    elseif (c >= epsilon .and. lab > L_upper) then
       print*, 'lab is above upper bound in UlUc, lab = ', lab
       pause
       UlUc = -1.0 * b * (c / (Le - L_upper))**xchi
    elseif (c < epsilon .and. lab < epsilon) then
       print*, 'c is below lower bound in UlUc, c = ', c
       print*, 'lab is below lower bound in UlUc, lab = ', lab
       pause
       UlUc = -1.0 * b * (epsilon / (Le - epsilon))**xchi
    elseif (c < epsilon .and. lab > L_upper) then
       print*, 'c is below lower bound in UlUc, c = ', c
       print*, 'lab is above upper bound in UlUc, lab = ', lab
       pause
       UlUc = -1.0 * b * (epsilon / (Le - L_upper))**xchi
    else
       print*, 'c is below lower bound in the function UlUc, c = ', c	
       pause
       UlUc = -1.0 * b * (epsilon / (Le - lab))**xchi
    end if
    
  end function UlUc

  !------------------------------------------------------------------------------------
  ! function F
  !
  !	Production Function - Takes country-specific capital stock and 
  !	technology shock and returns output.
  !
  ! Inputs:	capital -- real, capital level
  !             lab -- real, labor supply
  !	        sh -- real, technological shock   
  ! Returns:	real, output
  !
  !------------------------------------------------------------------------------------
  
  function F(capital,lab,sh)
    implicit none
    real(prec) :: F
    real(prec), intent(in) :: capital,lab,sh
    real(prec) :: epsilon
    
    epsilon = (10.0)**(-6)
    
    if (capital >= epsilon .and. lab >= epsilon) then
       F =  A_tfp * exp(sh) * (alpha * capital**muh + (1.0-alpha) * lab**muh)**(1.0/muh)
    elseif (capital < epsilon .and. lab >= epsilon) then
       print*, "Error - Capital Stock is negative in F, cap = ", capital	
       pause
       F = 0
    elseif (capital >= epsilon .and. lab < epsilon) then
       print*, "Error - Labor Supply is negative in F, lab = ", lab
       pause
       F = 0
    else
       print*, "Error - Capital Stock is negative in F, cap = ", capital
       print*, "Error - Labor Supply is negative in F, lab = ", lab	
       pause
       F = 0
    end if
  end function F
  
  !----------------------------------------------------------------------------
  ! function Fk
  !
  !	Marginal Product of Capital - Takes country specific capital stock and 
  !	technology shock and returns mpk.
  !
  ! Inputs:	capital -- real, capital level
  !             lab -- real, labor supply
  !		sh -- real, technological shock   
  ! Returns:	real, mpk
  !
  !----------------------------------------------------------------------------
  
  function Fk(capital,lab,sh)
    implicit none
    real(prec) :: Fk
    real(prec), intent(in) :: capital,lab,sh
    real(prec) :: epsilon
    
    epsilon = (10.0)**(-6)
    
    if (capital >= epsilon .and. lab >= epsilon) then
       Fk = alpha * A_tfp * exp(sh) * (F(capital,lab,sh)/A_tfp/exp(sh)/capital)**(1.0 - muh)
    elseif (capital < epsilon .and. lab >= epsilon) then
       print*, "Error - Capital Stock is negative in Fk, cap = ", capital  
       pause
       Fk = alpha * A_tfp * exp(sh) * (F(epsilon,lab,sh)/A_tfp/exp(sh)/epsilon)**(1.0 - muh)
    elseif (capital >= epsilon .and. lab < epsilon) then
       print*, "Error - Labor Supply is negative in Fk, lab = ", lab  
       pause
       Fk = alpha * A_tfp * exp(sh) * (F(capital,epsilon,sh)/A_tfp/exp(sh)/capital)**(1.0 - muh)
    else
       print*, "Error - Capital Stock is negative in Fk, cap = ", capital  
       print*, "Error - Labor Supply is negative in Fk, lab = ", lab  
       pause
       Fk = alpha * A_tfp * exp(sh) * (F(epsilon,epsilon,sh)/A_tfp/exp(sh)/epsilon)**(1.0 - muh)
    end if
    
  end function Fk
  
  !----------------------------------------------------------------------------
  ! function Fl
  !
  !	Marginal Product of Labor - Takes country specific capital stock and 
  !	technology shock and returns mpl.
  !
  ! Inputs:	capital -- real, capital level
  !             lab -- real, labor supply
  !		sh -- real, technological shock   
  ! Returns:	real, mpl
  !
  !----------------------------------------------------------------------------
  
  function Fl(capital,lab,sh)
    implicit none
    real(prec) :: Fl
    real(prec), intent(in) :: capital,lab,sh
    real(prec) :: epsilon
    
    epsilon = (10.0)**(-6)
    
    if (capital >= epsilon .and. lab >= epsilon) then
       Fl = (1.0-alpha) * A_tfp * exp(sh) * (F(capital,lab,sh)/A_tfp/exp(sh)/lab)**(1.0 - muh)
    elseif (capital < epsilon .and. lab >= epsilon) then
       print*, "Error - Capital Stock is negative in Fl, cap = ", capital  
       !pause
       Fl = (1.0-alpha) * A_tfp * exp(sh) * (F(epsilon,lab,sh)/A_tfp/exp(sh)/lab)**(1.0 - muh)
    elseif (capital >= epsilon .and. lab < epsilon) then
       print*, "Error - Labor Supply is negative in Fl, lab = ", lab  
       !pause
       Fl = (1.0-alpha) * A_tfp * exp(sh) * (F(capital,epsilon,sh)/A_tfp/exp(sh)/epsilon)**(1.0 - muh)
    else
       print*, "Error - Capital Stock is negative in Fl, cap = ", capital  
       print*, "Error - Labor Supply is negative in Fl, lab = ", lab  
       !pause
       Fl = (1.0-alpha) * A_tfp * exp(sh) * (F(epsilon,epsilon,sh)/A_tfp/exp(sh)/epsilon)**(1.0 - muh)
    end if
    
  end function Fl

end module Params
